#!/bin/bash

PROJECTURI=$1
UNAME=$2
PWORD=$3

PROJECTNAME=$(basename "$PROJECTURI")
PROJECTNAME="${PROJECTNAME%.*}"
PROJECTID=$(cat /proc/sys/kernel/random/uuid)

UPLOAD_TIMESTAMP=$(date --iso-8601=seconds)

LOG_FILENAME="wincanweb.log"

HCS12_STATUS_SENDING=1
HCS12_STATUS_SENT=4
HCS12_STATUS_FAILED_TO_SEND=5

# These are the return codes we can use
RET_SUCCESSFUL=0
RET_LOGIN_FAILED=1
RET_SERVICE_ERROR=2
RET_ERROR=3

function logToFile {
	echo "[$(date)] $1" >> $LOG_FILENAME
}

function debugMsg {
	echo "::INF::$1"
	logToFile "$1"
}

function exitAndSetStatus {
	echo $1 >> wincanweb.result
	
	if [[ $1 == $RET_SUCCESSFUL ]] 
	then
		echo "::EM::$HCS12_STATUS_SENT 100"
		debugMsg "SUCCESS"
	else
		echo "::EM::$HCS12_STATUS_FAILED_TO_SEND 100"
		debugMsg "FAILURE"
	fi
	
	sleep 1
	exit $1
}

function checkForLogin {
	if [ -z "$UNAME" ]
	then
		debugMsg "No username provided"
		exitAndSetStatus $RET_LOGIN_FAILED
	else
		if [ -z "$PWORD" ]
		then
			debugMsg "No password provided"
			exitAndSetStatus $RET_LOGIN_FAILED
		else
			./wincan-web-login-gen.sh $UNAME $PWORD
		fi
	fi
}

function init {
	echo "::EM::$HCS12_STATUS_SENDING 0"

	FILENAME=$(basename "$PROJECTURI")
	
	if [[ $FILENAME == *".wcx" ]] ; then
		debugMsg "WCX Project folder found"
	else
		debugMsg "WCX Project folder not found"
		exitAndSetStatus $RET_ERROR
	fi

	REMAININGSIZE=$(du -s "$PROJECTURI" | awk '{ print $1 }')
	STARTSIZE=$REMAININGSIZE

	if ((STARTSIZE==0)) ; then 
		debugMsg "Error: Empty folder"
		exitAndSetStatus $RET_ERROR
	fi
	
	debugMsg "DirectorySize: $STARTSIZE KB"
	echo "0" > wincanweb.progress
	echo "0" >> wincanweb.progress
}

function updateProgress {
	FILESIZE_UPLOADED=$1

	REMAININGSIZE=`expr $REMAININGSIZE - $FILESIZE_UPLOADED`
	debugMsg "Remaining: $REMAININGSIZE KB"
	
	PERCENT=$((99 - (REMAININGSIZE * 100 / STARTSIZE)))
	echo "$PERCENT" >> wincanweb.progress
	
	echo "::EM::$HCS12_STATUS_SENDING $PERCENT"
}

function endProgress {
	echo "100" >> wincanweb.progress
}

function jsonval {
    temp=`echo $json | sed 's/\\\\\//\//g' | sed 's/[{}]//g' | awk -v k="text" '{n=split($0,a,","); for (i=1; i<=n; i++) print a[i]}' | sed 's/\"\:\"/\|/g' | sed 's/[\,]/ /g' | sed 's/\"//g' | grep -w $prop`
    echo ${temp##*|}
}


# this function needs METHOD and APIURL setting
function generateWincanServiceHeaders {
	json=`curl -s -i -X GET http://52.48.54.7/time.php`
	
	if [[ $json == *"day"* ]] ; then
		RESULT=true
	else
		debugMsg "Get time failed"
		exitAndSetStatus $RET_ERROR
	fi

	prop='day'
	WEB_DAY=`jsonval`

	prop='date'
	WEB_DATE=`jsonval`

	prop='time'
	WEB_TIME=`jsonval`

	prop='timezone'
	WEB_TIMEZONE=`jsonval`

	CURRENT_TIME_STR="$WEB_DAY, $WEB_DATE $WEB_TIME $WEB_TIMEZONE"

	TOKEN=$(cat /proc/sys/kernel/random/uuid)

	REPRESENTATION="$METHOD\n$CURRENT_TIME_STR\n$TOKEN\nWinCanWeb\n$APIURL"

	json=`curl -s -i -X POST -d "data=$REPRESENTATION&secret=DC7C31B7-36F4-4EFA-984E-748BF05D9F9C" http://52.48.54.7/hash.php`
	
	if [[ $json == *"b64hmac"* ]] ; then
		RESULT=true
	else
		debugMsg "Get hash failed"
		exitAndSetStatus $RET_ERROR
	fi
	
	
	prop='b64hmac'
	SIGNATURE=`jsonval`
}

function flagFileForTranscoding () {
	METHOD="PUT"
	APIURL="api/job/putforeachpresets"
	generateWincanServiceHeaders

	rm wincanweb.content
	touch wincanweb.content

	echo '<PutFileRequest xmlns:i="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://schemas.datacontract.org/2004/07/CDLAB.WinCan.Service.SDK.Models.Request">' >> wincanweb.content
	echo "<BucketName>${BUCKETNAME}</BucketName><InputFileArray>$FILENAME</InputFileArray><OutputFilePath>Video/Sec/${FILENAME}</OutputFilePath>" >> wincanweb.content
	echo "<ProjectFolderName>${PROJECTNAME}</ProjectFolderName><Tasks><PutJobForEachPresetRequest>" >> wincanweb.content
	echo "<BucketName>${BUCKETNAME}</BucketName><FileKey>Video/Sec/${FILENAME}</FileKey>" >> wincanweb.content
	echo "<OutputKeyWithoutExtension>Video/Sec/web/${FILENAMENOEXT}</OutputKeyWithoutExtension></PutJobForEachPresetRequest></Tasks></PutFileRequest>" >> wincanweb.content
	
	rm wincanweb.content
	touch wincanweb.content
	echo '<PutJobForEachPresetRequest xmlns:i="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://schemas.datacontract.org/2004/07/CDLAB.WinCan.Service.SDK.Models.Request">' >> wincanweb.content
	echo "<BucketName>${BUCKETNAME}</BucketName><FileKey>Video/Sec/${FILENAME}</FileKey>" >> wincanweb.content
	echo "<OutputKeyWithoutExtension>Video/Sec/web/${FILENAMENOEXT}</OutputKeyWithoutExtension></PutJobForEachPresetRequest>" >> wincanweb.content
			
	debugMsg "Queue video for transcoding"
	
	json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/xml" -H "User-Agent: WinCanWeb" -H "Date: $CURRENT_TIME_STR" -H "Authorization: WinCanService $SIGNATURE" -H "X-WinCanService-MessageToken: $TOKEN" -H "X-WinCanService-ClientId: WinCanWeb" -X PUT --data @wincanweb.content https://minicam.web.wincan.com/wincanservice/api/job/putforeachpresets`
	
	if [[ $json == *"{}"* ]] ; then
		debugMsg "Transcoding started"
		RESULT=true
	else
		debugMsg "Transcoding failed"
		RESULT=false
	fi
}

function uploadFile {
	
	FILENAME=$(basename "$FILEURI")
	EXTENSION="${FILENAME##*.}"
	debugMsg "Uploading: $FILENAME"
	
	ISVIDEO=false

	rm wincanweb.content
	touch wincanweb.content
	# Base 64 encode the files and strip out all the newlines
	# Our CCU version of base64 doesnt support the -w 0 command so we pipe to 'tr' 
	BASE64FILE=`base64 "$FILEURI" | tr -d '\n'`

	echo "{'BucketName':'${BUCKETNAME}','InputFileArray':'" >> wincanweb.content
	echo $BASE64FILE >> wincanweb.content
	
	if [[ $EXTENSION == *"jpg"* ]] ; then
		echo "','OutputFilePath':'Picture/Sec/${FILENAME}','ProjectFolderName':'/','Tasks':[{},{}]}" >> wincanweb.content
	else
		echo "','OutputFilePath':'${FILENAME}','ProjectFolderName':'/','Tasks':[{},{}]}" >> wincanweb.content
	fi

	METHOD="PUT"
	APIURL="api/bucket/putfile"
	generateWincanServiceHeaders

	# let us now start the file upload
	json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/json" -H "User-Agent: WinCanWeb" -H "Date: $CURRENT_TIME_STR" -H "Authorization: WinCanService $SIGNATURE" -H "X-WinCanService-MessageToken: $TOKEN" -H "X-WinCanService-ClientId: WinCanWeb" -X PUT --data @wincanweb.content https://minicam.web.wincan.com/wincanservice/api/bucket/putfile`
	
	if [[ $json == *"VersionId"* ]] ; then
		debugMsg "file uploaded"
	else
		debugMsg "file failed"
		exitAndSetStatus $RET_ERROR
	fi
	
	updateProgress $(du -s "$FILEURI" | awk '{ print $1 }')
	
	# Check if this needs transcoding - retry once if necessary 
	if [ "$ISVIDEO" = true ] ; then
		sleep 5 # Small delay to let files assemble
		flagFileForTranscoding
		if [ "$RESULT" = false ] ; then
			debugMsg "Waiting to retry"
			sleep 30
			flagFileForTranscoding
			if [ "$RESULT" = false ] ; then
				debugMsg "ERROR - TRANSCODING HAS FAILED"
				#exitAndSetStatus $RET_ERROR
			fi
		fi
	fi
}

function uploadChunk () {
	CHUNKUPLOADID=$1
	CHUNKFILE=$2
	CHUNKNO=$3
	
	BASE64FILE=`base64 "${CHUNKFILE}" | tr -d '\n'`

	rm wincanweb.content
	touch wincanweb.content

	echo "{'MultipartUploadId':'${CHUNKUPLOADID}','InputFilePartArray':'" >> wincanweb.content
	echo $BASE64FILE >> wincanweb.content
	echo "','PartNumber':${CHUNKNO}}" >> wincanweb.content


	METHOD="PUT"
	APIURL="api/multipartupload/upload"
	generateWincanServiceHeaders

	debugMsg "Upload chunk $CHUNKNO"

	# upload the chunk 
	json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/json" -H "User-Agent: WinCanWeb" -H "Date: $CURRENT_TIME_STR" -H "Authorization: WinCanService $SIGNATURE" -H "X-WinCanService-MessageToken: $TOKEN" -H "X-WinCanService-ClientId: WinCanWeb" -X PUT --data @wincanweb.content https://minicam.web.wincan.com/wincanservice/api/multipartupload/upload`
	
	if [[ $json == *"DataChecksum"* ]] ; then
		debugMsg "chunk uploaded"
	else
		debugMsg "chunk failed"
		echo $json
		#rm -rf sp
		exitAndSetStatus $RET_ERROR
	fi
	
	updateProgress $(du -s "$CHUNKFILE" | awk '{ print $1 }')
}

function uploadMultipartFile {
	FILENAME=$(basename "$FILEURI")
	EXTENSION="${FILENAME##*.}"
	FILENAMENOEXT="${FILENAME%.*}"
	debugMsg "Uploading Multipart file: ${FILENAME}"
	
	ISVIDEO=false
	
	METHOD="PUT"
	APIURL="api/multipartupload/init"
	generateWincanServiceHeaders

	rm wincanweb.content
	touch wincanweb.content

	echo '<InitMultipartUploadRequest xmlns:i="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://schemas.datacontract.org/2004/07/CDLAB.WinCan.Service.SDK.Models.Request"><BucketName>' >> wincanweb.content
	echo $BUCKETNAME >> wincanweb.content
	echo '</BucketName><OutputFilePath>' >> wincanweb.content
	
	if [[ $EXTENSION == *"avi"* ]] ; then
		echo "Video/Sec/$FILENAME" >> wincanweb.content
		ISVIDEO=true
	elif [[ $EXTENSION == *"mp4"* ]] ; then
		echo "Video/Sec/$FILENAME" >> wincanweb.content
		ISVIDEO=true
	else
		echo $FILENAME >> wincanweb.content
	fi
	
	echo '</OutputFilePath></InitMultipartUploadRequest>' >> wincanweb.content

	# let us now start the file upload
	json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/xml" -H "User-Agent: WinCanWeb" -H "Date: $CURRENT_TIME_STR" -H "Authorization: WinCanService $SIGNATURE" -H "X-WinCanService-MessageToken: $TOKEN" -H "X-WinCanService-ClientId: WinCanWeb" -X PUT --data @wincanweb.content https://minicam.web.wincan.com/wincanservice/api/multipartupload/init`
	
	if [[ $json == *"MultipartUploadId"* ]] ; then
		RESULT=true
	else
		debugMsg "MultipartUploadId failed"
		exitAndSetStatus $RET_ERROR
	fi
	
	prop='MultipartUploadId'
	UPLOADID=`jsonval`

	# We now have an upload ID
	# Start uploading chunks
	
	##############################
	
	# holding area for biggest chunks
	rm -rf sp-large
	mkdir sp-large

	# holding area for sub chunks 
	rm -rf sp-sub
	mkdir sp-sub

	debugMsg "Splitting into large chunks"
	# 392MB was chosen as we will split again down to 4MB chunks but we cant exceed 99 chunks in 1 split 
	# as this is a limitation of the split command, this results in a max number of splits to 98
	# This makes our theoretical max size we can upload is 38808MB - bigger than our internal storage and 
	# also way bigger than FAT32 allows files to be
	split -d -b 392m "$FILEURI" sp-large/

	debugMsg  "sub dividing the large chunks"

	FOLDERNUM=0
	for entry in sp-large/*
	do
	  debugMsg "SPLIT $entry -> sp-sub/$FOLDERNUM/"
	  mkdir sp-sub/$FOLDERNUM
	  
	  # 4MB is the max filesize for wincan web
	  split -d -b 4m $entry sp-sub/$FOLDERNUM/
	  
	  # clean up the un-needed big chunk
	  rm $entry
	  
	  FOLDERNUM=$((FOLDERNUM + 1))
	done

	UPLOADNUM=0
	for entry in sp-sub/*
	do
		for subentry in $entry/*
		do
			debugMsg "UPLOAD $UPLOADID $subentry $UPLOADNUM"
			uploadChunk $UPLOADID $subentry $UPLOADNUM
			UPLOADNUM=$((UPLOADNUM + 1))
		done
	done

	# cleanup chunks
	rm -rf sp-large
	rm -rf sp-sub

	##############################
	# Finalise the file

	METHOD="PUT"
	APIURL="api/multipartupload/complete"
	generateWincanServiceHeaders

	rm wincanweb.content
	touch wincanweb.content

	echo '<CompleteMultipartUploadRequest xmlns:i="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://schemas.datacontract.org/2004/07/CDLAB.WinCan.Service.SDK.Models.Request"><CompleteAynchronously>true</CompleteAynchronously><MultipartUploadId>' >> wincanweb.content 
	echo $UPLOADID >> wincanweb.content 
	echo '</MultipartUploadId><ProjectFolderName>' >> wincanweb.content
	echo $PROJECTNAME >> wincanweb.content
	echo '</ProjectFolderName><Tasks><BaseTaskRequest /><BaseTaskRequest /></Tasks></CompleteMultipartUploadRequest>' >> wincanweb.content

	debugMsg "Upload finalised"	

	# finalise the file
	json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/xml" -H "User-Agent: WinCanWeb" -H "Date: $CURRENT_TIME_STR" -H "Authorization: WinCanService $SIGNATURE" -H "X-WinCanService-MessageToken: $TOKEN" -H "X-WinCanService-ClientId: WinCanWeb" -X PUT --data @wincanweb.content https://minicam.web.wincan.com/wincanservice/api/multipartupload/complete`

	if [[ $json == *"FilePk"* ]] ; then
		RESULT=true
	else
		debugMsg "finalise failed"
		exitAndSetStatus $RET_ERROR
	fi
	
	# Check if this needs transcoding - retry once if necessary 
	if [ "$ISVIDEO" = true ] ; then
		sleep 20 # Small delay to let files assemble
		flagFileForTranscoding
		if [ "$RESULT" = false ] ; then
			debugMsg "Waiting to retry"
			sleep 60
			flagFileForTranscoding
			if [ "$RESULT" = false ] ; then
				debugMsg "ERROR - TRANSCODING HAS FAILED"
				exitAndSetStatus $RET_ERROR
			fi
		fi
	fi
}

##################################################################################
#
# Main
#
##################################################################################


# Check if we have a log file and create
if [ ! -f $LOG_FILENAME ]; then
	touch $LOG_FILENAME
fi

debugMsg "Initiating WinCanWeb $PROJECTURI"

# Regenerate login file if needed
checkForLogin

# Init
init

# Check if we have a login file
if [ ! -f wincanweb.login ]; then
    debugMsg "No login credentials found"
	exitAndSetStatus $RET_LOGIN_FAILED
fi

# Login request
json=`curl --insecure -s -d "@wincanweb.login" -H "Content-Type: application/x-www-form-urlencoded" -X POST https://minicam.web.wincan.com/Token`

if [[ $json == *"access_token"* ]] ; then
	ERR_RESULT=false
else
	debugMsg "Login Error"
	exitAndSetStatus $RET_ERROR
fi


# Lets extract the info we need 
prop='access_token'
ACCESS_TOKEN=`jsonval`

if [ -z "$ACCESS_TOKEN" ]; then
    debugMsg "ACCESS_TOKEN not filled, login failed"
	exitAndSetStatus $RET_LOGIN_FAILED
fi

# Status request
#curl -s -i -H "Accept: application/json" -H "Content-Type: application/json" -X GET https://minicam.web.wincan.com/wincanservice/api/Status/get

# company wincan service - this needs access token only and gets us our secret key
json=`curl --insecure -s -i -H "Accept: application/json" -H "Authorization: Bearer ${ACCESS_TOKEN}" -X GET https://minicam.web.wincan.com/api/projects/companywincanservice`

if [[ $json == *"secretKey"* ]] ; then
	ERR_RESULT=false
else
	debugMsg "Token failed"
	exitAndSetStatus $RET_ERROR
fi

prop='secretKey'
SECRET_KEY=`jsonval`

if [ -z "$ACCESS_TOKEN" ]; then
    debugMsg "SECRET_KEY not filled, wincan issue"
	exitAndSetStatus $RET_SERVICE_ERROR
fi
# lets kick off an upload
json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/json" -H "Authorization: Bearer ${ACCESS_TOKEN}" -X POST -d "{'projectId':'${PROJECTID}','projectName':'${PROJECTNAME}','creationDate':'${UPLOAD_TIMESTAMP}','userName':'${UNAME}','hasPermissionsToOverrideAllProject':true, 'MiraculixImport':true}" https://minicam.web.wincan.com/api/uploadproject/init`

if [[ $json == *"bucketName"* ]] ; then
	ERR_RESULT=false
else
	debugMsg "Could not assign bucket"
	exitAndSetStatus $RET_ERROR
fi

prop='bucketName'
BUCKETNAME=`jsonval`

debugMsg "Assigned Bucket: $BUCKETNAME"

# check if our upload bucket exists - this needs full auth

METHOD="GET"
APIURL="api/bucket/${BUCKETNAME}"
generateWincanServiceHeaders

json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/json" -H "User-Agent: WinCanWeb" -H "Date: $CURRENT_TIME_STR" -H "Authorization: WinCanService $SIGNATURE" -H "X-WinCanService-MessageToken: $TOKEN" -H "X-WinCanService-ClientId: WinCanWeb" -X GET https://minicam.web.wincan.com/wincanservice/api/bucket/${BUCKETNAME}`

if [[ $json == *"Exist"* ]] ; then
	ERR_RESULT=false
else
	debugMsg "Could not verify bucket"
	exitAndSetStatus $RET_ERROR
fi

prop='Exist'
BUCKETEXIST=`jsonval`

# if the bucket isnt available the lets make a new one
if [ $BUCKETEXIST = "Exist:false" ] ; then
	debugMsg "Need to create bucket"
	METHOD="PUT"
	APIURL="api/bucket/$BUCKETNAME"
	generateWincanServiceHeaders
	json=`curl --insecure -s -i -H "Accept: application/json" -H "User-Agent: WinCanWeb" -H "Date: $CURRENT_TIME_STR" -H "Authorization: WinCanService $SIGNATURE" -H "X-WinCanService-MessageToken: $TOKEN" -H "X-WinCanService-ClientId: WinCanWeb" -X PUT -d "BucketName=${BUCKETNAME}" https://minicam.web.wincan.com/wincanservice/api/bucket/${BUCKETNAME}`
	
	if [[ $json == *"Exist"* ]] ; then
		ERR_RESULT=false
	else
		debugMsg "Could not create bucket"
		exitAndSetStatus $RET_ERROR
	fi
fi

# Cycle through all the files we need to upload
# If larger than 4 MB upload using multipart upload
for entry in "$PROJECTURI"/*
do
	 # by default this gives size in KB
	SIZE=$(du -s "$entry" | awk '{ print $1 }')

	FILEURI=$entry
	FILENAME=$(basename "$FILEURI")
	EXTENSION="${FILENAME##*.}"
	
	# Make sure to upload all video files as multipart
	# for some reason transcoding wont work without this
	
	if [[ $EXTENSION == *"mp4"* ]] ; then
		uploadMultipartFile
	elif [[ $EXTENSION == *"avi"* ]] ; then
		uploadMultipartFile
	elif ((SIZE<4000)) ; then 
		uploadFile
	else
		uploadMultipartFile
	fi
done

# lets finish off a complete project upload
json=`curl --insecure -s -i -H "Accept: application/json" -H "Content-Type: application/json" -H "Authorization: Bearer ${ACCESS_TOKEN}" -X POST -d "{'projectId':'${PROJECTID}', 'MiraculixImport':true}" https://minicam.web.wincan.com/api/uploadproject/uploaded`

if [[ $json == *"true"* ]] ; then
	ERR_RESULT=false
else
	debugMsg "Project failed to finalise"
	exitAndSetStatus $RET_ERROR
fi

endProgress

exitAndSetStatus $RET_SUCCESSFUL

